import pandas as pd
import numpy as np
import asyncio
from app.analysis import FootballAnalyzer
from app.database import engine, Base, SessionLocal
from app.strategy import BettingStrategy

async def run_backtest(initial_bankroll=1000.0, num_games=100):
    print(f"--- Starting Backtest (Last {num_games} games) ---")
    
    async with SessionLocal() as db:
        analyzer = FootballAnalyzer(db)
        
        # 1. Fetch historical matches
        all_matches = await analyzer.get_all_matches()
        if all_matches.empty:
            print("No matches found in DB.")
            return

        # 2. Split into 'past' (training) and 'future' (testing)
        if len(all_matches) < 20:
            print("Not enough matches for backtesting.")
            return

        test_set = all_matches.iloc[-num_games:].copy()
        
        current_bankroll = initial_bankroll
        bets_placed = 0
        bets_won = 0
        
        print(f"Initial Bankroll: ${current_bankroll:.2f}")
        
        test_1x2 = 1
        test_o15 = 0
        test_o25 = 0
        test_o35 = 0
        test_btts = 0
        
        for idx, row in test_set.iterrows():
            try:
                # Simulate prediction "before" the match happened
                pred = await analyzer.analyze_match(row['time_casa'], row['time_fora'])
                
                total_goals = row['placar_casa'] + row['placar_fora']
                btts_happened = (row['placar_casa'] > 0) and (row['placar_fora'] > 0)
                
                # --- Market 1: Winner 1x2 ---
                rec = pred.winner_1x2
                if ((rec.home_probability >= 0.5 and rec.best_bet == "HOME") or (rec.away_probability >= 0.35 and rec.best_bet == "AWAY") or rec.draw_probability >= 0.3 or rec.best_bet == "DRAW") and test_1x2 == 1:
                    actual_result = "HOME" if row['placar_casa'] > row['placar_fora'] else \
                                    "AWAY" if row['placar_fora'] > row['placar_casa'] else "DRAW"
                    
                    stake_pct = rec.kelly_stake_pct / 100 if rec.kelly_stake_pct else 0.05
                    stake = current_bankroll * stake_pct
                    
                    # Simulated Odds
                    prob_key = f"{rec.best_bet.lower()}_probability"
                    prob = getattr(rec, prob_key)
                    
                    mult = 1.15 if rec.best_bet == "HOME" else 0.95 if rec.best_bet == "AWAY" else 1
                    
                    bookie_odds = (1.0 / prob) * mult
                    if bookie_odds >= 0.2 :
                        if rec.best_bet == actual_result:
                            profit = stake * (bookie_odds - 1)
                            current_bankroll += profit
                            bets_won += 1
                            res_str = "WIN"
                        else:
                            current_bankroll -= stake
                            res_str = "LOSS"
                        
                        bets_placed += 1
                        print(f"[1x2] {row['time_casa']} vs {row['time_fora']} | Bet: {rec.best_bet} | Stake: {stake:.2f} | Odds: {bookie_odds:.2f} | Res: {res_str}")
                
                # --- Market 2: Over 1.5 ---
                if pred.over_1_5.probability >= 0.75 and test_o15 == 1:
                    stake_pct = pred.over_1_5.kelly_stake_pct / 100 if pred.over_1_5.kelly_stake_pct else 0.05
                    stake = current_bankroll * stake_pct
                    odds = (1.0 / pred.over_1_5.probability) * 1.2 # Standard over/under odds
                    
                    if total_goals > 1.5:
                        profit = stake * (odds - 1)
                        current_bankroll += profit
                        bets_won += 1
                        res_str = "WIN"
                    else:
                        current_bankroll -= stake
                        res_str = "LOSS"
                    bets_placed += 1
                    print(f"[O1.5] {row['time_casa']} vs {row['time_fora']} | Bet: OVER 1.5 | Stake: {stake:.2f} | Odds: {odds:.2f} | Res: {res_str}")
                
                # --- Market 3: Over 2.5 ---
                if pred.over_2_5.probability >= 0.6 and test_o25 == 1:
                    stake_pct = pred.over_2_5.kelly_stake_pct / 100 if pred.over_2_5.kelly_stake_pct else 0.05
                    stake = current_bankroll * stake_pct
                    odds = (1.0 / pred.over_2_5.probability) * 1.4 # Standard over/under odds
                    
                    if total_goals > 2.5:
                        profit = stake * (odds - 1)
                        current_bankroll += profit
                        bets_won += 1
                        res_str = "WIN"
                    else:
                        current_bankroll -= stake
                        res_str = "LOSS"
                    bets_placed += 1
                    print(f"[O2.5] {row['time_casa']} vs {row['time_fora']} | Bet: OVER 2.5 | Stake: {stake:.2f} | Odds: {odds:.2f} | Res: {res_str}")
                
                # --- Market 4: Over 3.5 ---
                if pred.over_3_5.probability >= 0.45 and test_o35 == 1:
                    stake_pct = pred.over_3_5.kelly_stake_pct / 100 if pred.over_3_5.kelly_stake_pct else 0.05
                    stake = current_bankroll * stake_pct
                    odds = (1.0 / pred.over_3_5.probability) * 2 # Standard over/under odds
                    
                    if total_goals > 3.5:
                        profit = stake * (odds - 1)
                        current_bankroll += profit
                        bets_won += 1
                        res_str = "WIN"
                    else:
                        current_bankroll -= stake
                        res_str = "LOSS"
                    bets_placed += 1
                    print(f"[O3.5] {row['time_casa']} vs {row['time_fora']} | Bet: OVER 3.5 | Stake: {stake:.2f} | Odds: {odds:.2f} | Res: {res_str}")

                # --- Market 4: BTTS ---
                if pred.btts.probability >= 0.55 and test_btts == 1:
                    stake_pct = pred.btts.kelly_stake_pct / 100 if pred.btts.kelly_stake_pct else 0.05
                    stake = current_bankroll * stake_pct
                    odds = (1.0 / pred.btts.probability) * 1.3
                    
                    if btts_happened:
                        profit = stake * (odds - 1)
                        current_bankroll += profit
                        bets_won += 1
                        res_str = "WIN"
                    else:
                        current_bankroll -= stake
                        res_str = "LOSS"
                    bets_placed += 1
                    print(f"[BTTS] {row['time_casa']} vs {row['time_fora']} | Bet: YES | Stake: {stake:.2f} | Odds: {odds:.2f} | Res: {res_str}")

            except Exception as e:
                print(f"Error processing match: {e}")
                
        print(f"--- Backtest Finished ---")

        print(f"--- Backtest Finished ---")
        print(f"Bets Placed: {bets_placed}")
        print(f"Wins: {bets_won}")
        print(f"Final Bankroll: ${current_bankroll:.2f}")
        roi = ((current_bankroll - initial_bankroll) / initial_bankroll) * 100
        print(f"ROI: {roi:.2f}%")

if __name__ == "__main__":
    loop = asyncio.new_event_loop()
    asyncio.set_event_loop(loop)
    loop.run_until_complete(run_backtest())
