import pandas as pd
import numpy as np
from scipy.stats import poisson
from sqlalchemy import select, or_, desc
from sqlalchemy.ext.asyncio import AsyncSession
from app.models import Match
from app.schemas import MatchPrediction, MarketPrediction
from typing import List, Tuple

from app.feature_engineering import FeatureEngineer
from app.ml_engine import MLEngine
from app.strategy import BettingStrategy

class FootballAnalyzer:
    def __init__(self, db: AsyncSession):
        self.db = db
        self.feature_engineer = FeatureEngineer()
        self.ml_engine = MLEngine()
        self.strategy = BettingStrategy()

    async def get_team_history(self, team_name: str, limit: int = 20) -> pd.DataFrame:
        """Fetches the last N games for a specific team."""
        query = select(Match).where(
            or_(Match.time_casa == team_name, Match.time_fora == team_name)
        ).order_by(desc(Match.id_jogo)).limit(limit)
        
        result = await self.db.execute(query)
        matches = result.scalars().all()
        
        if not matches:
            return pd.DataFrame()

        data = []
        for m in matches:
            data.append({
                "id_jogo": m.id_jogo,
                "time_casa": m.time_casa,
                "time_fora": m.time_fora,
                "placar_casa": m.placar_casa,
                "placar_fora": m.placar_fora,
                "data_jogo": m.data_jogo
            })
        return pd.DataFrame(data)

    def calculate_metrics(self, df: pd.DataFrame, team: str) -> dict:
        """Calculates form metrics: Avg Goals Scored/Conceded."""
        if df.empty:
            return {"avg_goals_scored": 1.0, "avg_goals_conceded": 1.0} # Default values

        # Filter for games where team was Home vs Away
        home_games = df[df['time_casa'] == team]
        away_games = df[df['time_fora'] == team]

        goals_scored = home_games['placar_casa'].sum() + away_games['placar_fora'].sum()
        goals_conceded = home_games['placar_fora'].sum() + away_games['placar_casa'].sum()
        total_games = len(df)

        return {
            "avg_goals_scored": goals_scored / total_games if total_games > 0 else 0,
            "avg_goals_conceded": goals_conceded / total_games if total_games > 0 else 0,
            "recent_form": df.head(5) # Just for debug/logging
        }

    def predict_poisson(self, home_avg_scored, home_avg_conceded, away_avg_scored, away_avg_conceded) -> dict:
        """
        Uses Poisson distribution to calculate probabilities for exact scores.
        Lambda for Home = (Home Avg Scored + Away Avg Conceded) / 2
        Lambda for Away = (Away Avg Scored + Home Avg Conceded) / 2
        
        Includes Dixon-Coles adjustment for low-scoring draws.
        """
        # Weighted average for expected goals (lambda)
        # We can add league averages here for better accuracy if available
        lambda_home = (home_avg_scored * 0.6) + (away_avg_conceded * 0.4) 
        lambda_away = (away_avg_scored * 0.6) + (home_avg_conceded * 0.4)

        # Max goals to calculate probability for (e.g., up to 6 goals)
        max_goals = 6
        score_probs = np.zeros((max_goals + 1, max_goals + 1))

        for h in range(max_goals + 1):
            for a in range(max_goals + 1):
                prob = poisson.pmf(h, lambda_home) * poisson.pmf(a, lambda_away)
                
                # Dixon-Coles Adjustment
                # Usually rho is estimated, but a small positive rho (e.g. 0.13) corrects for the underestimation of 0-0 and 1-1
                rho = -0.13 # Standard value often used in literature, though it varies by league
                
                correction = 1.0
                if h == 0 and a == 0:
                    correction = 1.0 - (lambda_home * lambda_away * rho)
                elif h == 0 and a == 1:
                    correction = 1.0 + (lambda_home * rho)
                elif h == 1 and a == 0:
                    correction = 1.0 + (lambda_away * rho)
                elif h == 1 and a == 1:
                    correction = 1.0 - rho
                
                score_probs[h][a] = prob * correction

        # Renormalize probabilities to sum to 1.0
        total_prob = np.sum(score_probs)
        if total_prob > 0:
            score_probs = score_probs / total_prob

        return score_probs, lambda_home, lambda_away

    async def get_all_matches(self) -> pd.DataFrame:
        """Fetches all matches for training."""
        query = select(Match).order_by(Match.data_jogo)
        result = await self.db.execute(query)
        matches = result.scalars().all()
        
        data = []
        for m in matches:
            data.append({
                "id_jogo": m.id_jogo,
                "time_casa": m.time_casa,
                "time_fora": m.time_fora,
                "placar_casa": m.placar_casa,
                "placar_fora": m.placar_fora,
                "data_jogo": m.data_jogo
            })
        return pd.DataFrame(data)

    async def analyze_match(self, time_casa: str, time_fora: str) -> MatchPrediction:
        # 1. Fetch Data for Feature Engineering (need global context for accurate features)
        # Ideally, we should cache this dataframe or update it incrementally
        all_matches = await self.get_all_matches()
        
        # 2. Generate Features for the upcoming match
        features = self.feature_engineer.get_features_for_match(time_casa, time_fora, all_matches)
        
        # 3. Predict using ML if available, else fallback to simple stats
        lambda_home = 1.0
        lambda_away = 1.0
        btts_prob_ml = 0.5
        
        use_ml = False
        if features is not None and self.ml_engine.model_home is not None:
            try:
                preds = self.ml_engine.predict(features)
                lambda_home = preds['expected_home_goals']
                lambda_away = preds['expected_away_goals']
                btts_prob_ml = preds['btts_prob']
                use_ml = True
            except Exception as e:
                print(f"ML Prediction failed: {e}. Falling back to simple stats.")
        
        if not use_ml:
             # Fallback: Simple Average Logic
            hist_home = await self.get_team_history(time_casa, limit=20)
            hist_away = await self.get_team_history(time_fora, limit=20)
            metrics_home = self.calculate_metrics(hist_home, time_casa)
            metrics_away = self.calculate_metrics(hist_away, time_fora)
            
            lambda_home = (metrics_home['avg_goals_scored'] * 0.6) + (metrics_away['avg_goals_conceded'] * 0.4) 
            lambda_away = (metrics_away['avg_goals_scored'] * 0.6) + (metrics_home['avg_goals_conceded'] * 0.4)

        # 4. Poisson Model (Used for both ML and Simple approach to get score matrix)
        # Max goals to calculate probability for
        max_goals = 6
        score_probs = np.zeros((max_goals + 1, max_goals + 1))

        for h in range(max_goals + 1):
            for a in range(max_goals + 1):
                prob = poisson.pmf(h, lambda_home) * poisson.pmf(a, lambda_away)
                score_probs[h][a] = prob

        # 5. Aggregate Probabilities
        prob_home_win = np.sum(np.tril(score_probs, -1))
        prob_draw = np.sum(np.diag(score_probs))
        prob_away_win = np.sum(np.triu(score_probs, 1))

        prob_over_1_5 = 1 - np.sum(score_probs[0:2, 0:2]) + score_probs[1,1] # Approximation correction or simple sum
        # Correct summation for Over markets:
        prob_under_1_5 = score_probs[0,0] + score_probs[1,0] + score_probs[0,1]
        prob_over_1_5 = 1 - prob_under_1_5
        
        prob_under_2_5 = prob_under_1_5 + score_probs[2,0] + score_probs[0,2] + score_probs[1,1]
        prob_over_2_5 = 1 - prob_under_2_5
        
        prob_under_3_5 = prob_under_2_5 + score_probs[3,0] + score_probs[0,3] + score_probs[2,1] + score_probs[1,2]
        prob_over_3_5 = 1 - prob_under_3_5

        prob_btts = 1 - (np.sum(score_probs[0, :]) + np.sum(score_probs[:, 0]) - score_probs[0,0])

        # 5. Determine "Best Bet" (Simple Threshold Logic for now - Model would go here)
        # In a real scenario, compare these probs with Odds to find Value.
        # Here we just use high confidence thresholds.
        
        def evaluate_bet(prob, threshold=0.65):
            return {
                "probability": round(prob, 4),
                "should_bet": prob > threshold,
                "confidence": "High" if prob > 0.8 else ("Medium" if prob > threshold else "Low")
            }

        # 6. Correct Score Probabilities
        correct_score_probabilities = []
        for h in range(max_goals + 1):
            for a in range(max_goals + 1):
                prob = score_probs[h][a]
                if prob > 0.01: # Filter out very low probabilities to keep response clean
                    correct_score_probabilities.append({
                        "score": f"{h}-{a}",
                        "probability": round(float(prob), 4)
                    })
        
        # Sort by probability descending
        correct_score_probabilities.sort(key=lambda x: x['probability'], reverse=True)

        # 7. Winner 1x2 Logic
        winner_probs = {
            "HOME": prob_home_win,
            "DRAW": prob_draw,
            "AWAY": prob_away_win
        }
        best_bet_1x2 = max(winner_probs, key=winner_probs.get)
        confidence_1x2 = "High" if winner_probs[best_bet_1x2] > 0.6 else "Medium" if winner_probs[best_bet_1x2] > 0.45 else "Low"
        
        # Strategy for 1x2 (Assuming some default odds just to show functionality)
        # In a real scenario, you'd pass the actual bookmaker odds here
        # For now, we simulate odds that are slightly worse than fair odds (bookie margin)
        simulated_odds = 1.0 / winner_probs[best_bet_1x2] * 0.90 # 10% margin
        strat_1x2 = self.strategy.get_recommendation(winner_probs[best_bet_1x2], simulated_odds)

        # Strategy for Markets (using 1.90 as generic odds for Over/Under usually)
        # We can make this more dynamic later
        
        def evaluate_bet_with_strategy(prob, threshold=0.65):
            # Calculate fair odds (suggested odds) based on probability
            # Fair Odds = 1 / Probability
            suggested_odds = 1 / prob if prob > 0 else 0
            
            rec = self.strategy.get_recommendation(prob, 1.90) # Assume 1.90 odds for even money markets
            return {
                "probability": round(prob, 4),
                "should_bet": prob > threshold, # Only recommend based on raw probability for now unless we have real odds
                "confidence": "High" if prob > 0.8 else ("Medium" if prob > threshold else "Low"),
                "expected_value": rec['expected_value'],
                "kelly_stake_pct": rec['kelly_stake_pct'],
                "suggested_odds": round(suggested_odds, 2)
            }

        # Most likely score
        max_idx = np.unravel_index(np.argmax(score_probs, axis=None), score_probs.shape)
        most_likely_score = f"{max_idx[0]}-{max_idx[1]}"

        return MatchPrediction(
            time_casa=time_casa,
            time_fora=time_fora,
            winner_1x2={
                "home_probability": round(float(prob_home_win), 4),
                "draw_probability": round(float(prob_draw), 4),
                "away_probability": round(float(prob_away_win), 4),
                "best_bet": best_bet_1x2,
                "confidence": confidence_1x2,
                "value_bet": strat_1x2['recommendation'] in ["VALUE BET", "STRONG BET"],
                "expected_value": strat_1x2['expected_value'],
                "kelly_stake_pct": strat_1x2['kelly_stake_pct']
            },
            over_1_5=evaluate_bet_with_strategy(prob_over_1_5, 0.75),
            over_2_5=evaluate_bet_with_strategy(prob_over_2_5, 0.60),
            over_3_5=evaluate_bet_with_strategy(prob_over_3_5, 0.45),
            btts=evaluate_bet_with_strategy(prob_btts, 0.60),
            correct_score_probabilities=correct_score_probabilities,
            most_likely_score=most_likely_score
        )
